package com.wangxiaolu.promotion.common.excel;

import cn.hutool.core.date.DatePattern;
import com.wangxiaolu.promotion.exception.DataException;
import com.wangxiaolu.promotion.result.basedata.RCode;
import org.apache.poi.hssf.usermodel.HSSFWorkbook;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.xssf.usermodel.XSSFWorkbook;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.math.BigDecimal;
import java.util.*;


/**
 * @author : liqiulin
 * @date : 2024-08-16 15
 * @describe :读取Excel
 */
public class ReadExcelUtils {
    private Logger logger = LoggerFactory.getLogger(ReadExcelUtils.class);
    private Workbook wb;
    private Sheet sheet;
    private Row row;

    public ReadExcelUtils(String filepath) {
        if (filepath == null) {
            throw new DataException(RCode.READ_FILE_PATH_NULL_ERROR);
        }
        String ext = filepath.substring(filepath.lastIndexOf("."));
        try {
            InputStream is = new FileInputStream(filepath);
            if (".xls".equals(ext)) {
                wb = new HSSFWorkbook(is);
            } else if (".xlsx".equals(ext)) {
                wb = new XSSFWorkbook(is);
            } else {
                throw new DataException(RCode.EXCEL_IS_XLS_OR_XLSX_FILE);
            }
        } catch (FileNotFoundException e) {
            logger.error("FileNotFoundException", e);
            throw new DataException(RCode.READ_EXCEL_NULL_ERROR);
        } catch (IOException e) {
            logger.error("IOException", e);
            throw new DataException(RCode.READ_EXCEL_NULL_ERROR);
        }
    }

    /**
     * 读取Excel表格表头的内容
     *
     * @return String 表头内容的数组
     * @author zengwendong
     */
    public String[] readTitle() {
        if (wb == null) {
            throw new DataException("Workbook对象为空！");
        }
        sheet = wb.getSheetAt(0);
        row = sheet.getRow(0);
        // 标题总列数
        int colNum = row.getPhysicalNumberOfCells();
        System.out.println("colNum:" + colNum);
        String[] title = new String[colNum];
        for (int i = 0; i < colNum; i++) {
            // title[i] = getStringCellValue(row.getCell((short) i));
            title[i] = row.getCell(i).getStringCellValue();
        }
        return title;
    }

    /**
     * 读取Excel数据内容
     *
     * @return Map 包含单元格数据内容的Map对象
     * @author zengwendong
     */
    public Map<Integer, List<Object>> readContent(){
        if (wb == null) {
            throw new DataException("Workbook对象为空！");
        }

        sheet = wb.getSheetAt(0);
        int rowNum = sheet.getLastRowNum();
        Map<Integer, List<Object>> content = new HashMap<>(rowNum * 2);

        row = sheet.getRow(0);
        int colNum = row.getPhysicalNumberOfCells();

        // 正文内容应该从第二行开始,第一行为表头的标题
        for (int i = 1; i <= rowNum; i++) {
            row = sheet.getRow(i);
            int j = 0;

            List<Object> cellValues = new ArrayList<>(rowNum * 2);
            while (j < colNum) {
                Object obj = getCellFormatValue(row.getCell(j));
                cellValues.add(obj);
                j++;
            }

            // 如果row没有值，则停止解析
            Set vset = new HashSet(cellValues);
            if (vset.size() <= 1){
                return content;
            }

            content.put(i, cellValues);
        }
        return content;
    }

    /**
     * 根据Cell类型设置数据
     *
     * @param cell
     * @return
     * @author zengwendong
     */
    private Object getCellFormatValue(Cell cell) {
        Object cellvalue = "";
        //判断是否为null或空串
        if (cell == null || cell.toString().trim().equals("")) {
            return "";
        }

        CellType cellType = cell.getCellType();

        switch (cellType) {
            case STRING:
                cellvalue = cell.getStringCellValue();
                break;
            case NUMERIC:
                if (DateUtil.isCellDateFormatted(cell)) {
                    cellvalue = cell.getDateCellValue();
                } else {
                    BigDecimal number = new BigDecimal(cell.getNumericCellValue());
                    cellvalue = number.toPlainString();
                }
                break;
            default:
                return "";
        }
        return cellvalue;
    }
}